﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public class MapProviderServiceToCrm : MapperRelatedBase
    {
        private readonly MapCareSiteToCrm _careSiteMapper;

        public MapProviderServiceToCrm(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper, MapCareSiteToCrm careSiteMapper)
            : base(ppmsContextHelper, ppmsHelper)
        {
            _careSiteMapper = careSiteMapper;
        }

        public async Task<ppms_providerservice> MapInsert(ProviderService service, Account provider)
        {
            // Map entity
            var entity = new ppms_providerservice { Id = Guid.NewGuid() };

            // Specialty
            if (!string.IsNullOrEmpty(service.CodedSpecialty))
            {
                var specialty = GetTaxonomy(service.CodedSpecialty);
                if (specialty != null) entity.ppms_specialty = specialty;
            }

            // Affiliation
            entity.ppms_network = new EntityReference("ppms_vaprovidernetwork", NetworkId);
            
            // Care site
            if (service.Location != null) await SetCareSite(service.Location, entity);
            
            // Station Number
            //if (!string.IsNullOrEmpty(service.StationNumber))
            //{
            //    var careSite = await GetCareSiteByStationNumber(service.StationNumber);
                
            //    if (careSite != null) entity.ppms_caresite = entity.ToEntityReference();
            //}

            // Organization
            if (!string.IsNullOrEmpty(service.OrganizationNpi))
            {
                var org = await GetProviderByNpi(service.OrganizationNpi);
                if (org != null) entity.ppms_Organization = org.ToEntityReference();
            }
            
            if (!string.IsNullOrEmpty(service.Relationship))
            {
                // Relationship
                var relationship = GetVaProviderRelationship(service.Relationship);
                if (relationship != null)
                {
                    entity.ppms_VAProviderRelationshipId = relationship;
                }
            }

            // Set station number
            if (!string.IsNullOrEmpty(service.StationNumber))
            {
                var facility = await PpmsHelper.GetFacilityByStationNumber(service.StationNumber);
                if (facility != null) entity.ppms_Facility = facility;
            }

            return entity;
        }

        public async Task<ppms_providerservice> MapUpdate(ProviderService entity, Account provider)
        {
            if (string.IsNullOrEmpty(entity.CorrelationId)) return await MapInsert(entity, provider);

            // get entity
            var crmEntity = GetCrmEntity(provider, entity.CorrelationId);

            // matching account not found, insert new record
            if (crmEntity == null) throw new PpmsServiceException($"A record with the provided correlation ID ({entity.CorrelationId} could not be found.", "ProviderService");

            var newEntity = new ppms_providerservice
            {
                Id = crmEntity.Id,
                ppms_name = crmEntity.ppms_name
            };

            // Map fields
            if (!string.IsNullOrEmpty(entity.CodedSpecialty))
            {
                var specialty = GetTaxonomy(entity.CodedSpecialty);
                if (specialty != null) newEntity.ppms_specialty = specialty;
            }

            if (entity.Location != null) await SetCareSite(entity.Location, newEntity);
            else if (!string.IsNullOrEmpty(entity.StationNumber)) await SetCareSite(crmEntity, entity.StationNumber);

            if (!string.IsNullOrEmpty(entity.Relationship))
            {
                // Relationship
                var relationship = GetVaProviderRelationship(entity.Relationship);
                if (relationship != null)
                {
                    crmEntity.ppms_VAProviderRelationshipId = relationship;
                }
            }

            // Set station number
            if (!string.IsNullOrEmpty(entity.StationNumber))
            {
                var facility = await PpmsHelper.GetFacilityByStationNumber(entity.StationNumber);
                if (facility != null) crmEntity.ppms_Facility = facility;
            }

            return newEntity;
        }

        private async Task<ppms_caresite> SetCareSite(CareSite site, ppms_providerservice targetService)
        {
            // determine if care site exists
            ppms_caresite entity;
            if (string.IsNullOrEmpty(site.CorrelationId)) entity = await CheckForExistingCareSite(site);
            else
            {
                using (var context = await PpmsContextHelper.GetContextAsync())
                {
                    entity = context.ppms_caresiteSet.FirstOrDefault(a => a.Id == new Guid(site.CorrelationId));
                }
            }

            if (entity == null)
            {
                // Set reference values for child mapper
                _careSiteMapper.References = References;

                // Care site not found, map as insert
                entity = await _careSiteMapper.MapInsert(site);

                if (entity != null)
                {
                    // Set relationship to service
                    targetService.ppms_caresite_providerservice_caresite = entity;
                    
                    // Add to added entity list for future comparisons
                    AddModifiedEntity(entity);
                }
            }
            else targetService.ppms_caresite = entity.ToEntityReference();

            return entity;
        }

        private async Task<ppms_caresite> SetCareSite(ppms_providerservice entity, string stationNumber)
        {
            ppms_caresite careSite = null;

            // Station Number
            if (!string.IsNullOrEmpty(stationNumber))
            {
                careSite = await GetCareSiteByStationNumber(stationNumber);
                if (careSite != null)
                {
                    entity.ppms_caresite = entity.ToEntityReference();
                }
            }

            return careSite;
        }

        private static ppms_providerservice GetCrmEntity(Account provider, string id)
        {
            var list = provider.ppms_account_ppms_providerservice;
            if (list == null) return null;

            var ppmsProviderservices = list as ppms_providerservice[] ?? list.ToArray();

            return ppmsProviderservices.Any() ? ppmsProviderservices.FirstOrDefault(x => x.Id.ToString() == id) : null;
        }

        private async Task<ppms_caresite> CheckForExistingCareSite(CareSite careSite)
        {
            ppms_caresite match = null;
            var address = careSite.SiteAddress;

            if (address != null)
            {
                // Check for match in current batch of updates
                var activeList = ModifyEntityList;
                if (activeList != null && activeList.Any())
                {
                    var matchList = activeList.Where(x => x.GetAttributeValue<string>("ppms_name") == careSite.Name);
                    match = (ppms_caresite)matchList.FirstOrDefault(x => x.GetAttributeValue<string>("ppms_address_compositeid") == address.CompositeId);
                }

                // If no match found, search for existing entity
                if (match == null)
                {
                    using (var context = await PpmsContextHelper.GetContextAsync())
                    {
                        var matchList = context.ppms_caresiteSet.Where(c => c.ppms_name == careSite.Name);
                        match = matchList.FirstOrDefault(a => a.ppms_address_compositeid.Contains(address.CompositeId));
                    }
                }
            }

            // No match found
            return match;
        }

        private async Task<ppms_caresite> GetCareSiteByStationNumber(string stationNumber)
        {
            using (var context = await PpmsContextHelper.GetContextAsync())
            {
                return context.ppms_caresiteSet.FirstOrDefault(s => s.ppms_stationnumber == stationNumber);
            }
        }

        private EntityReference GetVaProviderRelationship(string relationshipType)
        {
            var result = References.VaProviderRelationships.FirstOrDefault(s => s.Value == relationshipType);
            if (result != null)
                return result.ToEntityReference("ppms_vaproviderrelationship");
            // Not found
            return null;
        }

        private static ProviderService ConvertEntity<T>(T entity)
        {
            return (ProviderService)Convert.ChangeType(entity, typeof(ProviderService));
        }

        public override async Task<Entity> MapUpdate<T>(T entity, Entity parent)
        {
            return await MapUpdate(ConvertEntity(entity), (Account)parent);
        }

        public override async Task<Entity> MapInsert<T>(T entity, Entity parent)
        {
            return await MapInsert(ConvertEntity(entity), (Account)parent);
        }

        public override async void AddChildrenToProvider(IList<Entity> entities, Entity parent)
        {
            if (entities == null) return;

            if (IsWithinContext)
            {
                foreach (var item in entities)
                {
                    Context.AddRelatedObject(parent, new Relationship("ppms_account_ppms_providerservice"), item);

                    var careSites = item.RelatedEntities.Values.FirstOrDefault();
                    if (careSites == null || careSites.TotalRecordCount <= 0) continue;

                    foreach (var entity in careSites.Entities)
                    {
                        Context.AddRelatedObject(item, new Relationship("ppms_caresite_providerservice_caresite"), entity);
                    }
                }
            }
            else
            {
                if (entities.Count <= 0) return;

                var account = (Account)parent;
                if (account != null) account.ppms_account_ppms_providerservice = ConvertEntityList<ppms_providerservice>(entities);
            }

            await Task.Run(() => { });
        }

        public override IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent)
        {
            if (entities == null || !entities.Any()) return null;

            // Check provider
            var provider = (Account)parent;
            if (provider == null) return null;

            var list = new List<T>();

            // Map schema entities for delete
            foreach (var item in entities)
            {
                var entity = ConvertEntity(item);
                var entity1 = entity;
                var matches = provider.ppms_account_ppms_providerservice.Where(p => p.Id == new Guid(entity1.CorrelationId));
                list.AddRange((IEnumerable<T>)matches);
            }

            return EntityDelete((IEnumerable<ppms_providerservice>)list);
        }
    }
}